from __future__ import absolute_import, print_function, unicode_literals
import json
import os

from .ableton_socket_server import AbletonSocketServer

class SocketManager:
    def __init__(self, parent):
        self.parent_instance = parent
        self.server = None
        self.CONFIG_FILE = os.path.expanduser("~/Documents/Cue2LiveConfig.json")
        self.DEFAULT_PORT = 6000
        self.ensure_config_exists()
        self.port = self.load_port()
        self.running_backup = False

    def start_server(self):
        if self.server is None:
            try:
                self.server = AbletonSocketServer(self.parent_instance, port=self.port)
                self.server.start_server()
                self.parent_instance._log("Socket server started on port {}".format(self.server.port))
            except Exception as e:
                self.parent_instance._log("Failed to start socket server: {}".format(e))
                    # Try fallback port 6500
                try:
                    self.running_backup = True
                    self.server = AbletonSocketServer(self.parent_instance, port=6500)
                    self.server.start_server()
                    self.parent_instance._log("Socket server started on fallback port 6500.")

                except Exception as e:
                    self.parent_instance._log("Failed to start socket server on fallback port 6500: {}".format(e))

    def stop_server(self):
        if self.server:
            self.server.stop_server()
            self.server = None
            self.parent_instance._log("Socket server stopped.")
            
    def ensure_config_exists(self):
        """Create config file with default values if it does not exist."""
        if not os.path.exists(self.CONFIG_FILE):
            self.save_port(self.DEFAULT_PORT)
            

    def save_port(self, new_port):
        """Save the port to the config file."""
        try:
            with open(self.CONFIG_FILE, "w") as f:
                json.dump({"port": new_port}, f, indent=4)
            self.parent_instance._log("Port updated to {}".format(new_port))
        except Exception as e:
            self.parent_instance._log("Error saving port: {}".format(e))
            
    def load_port(self):
        """Load the port from config file."""
        if os.path.exists(self.CONFIG_FILE):
            try:
                with open(self.CONFIG_FILE, "r") as f:
                    config = json.load(f)
                    return config.get("port", 6000)  # Default to 6000 if missing
            except ValueError:
                return 6000
        return 6000
    
