from __future__ import absolute_import, print_function, unicode_literals
import Live
from ableton.v2.control_surface import Component
from ableton.v2.control_surface.input_control_element import MIDI_NOTE_TYPE, MIDI_CC_TYPE



class MappingComponent(Component):
    def __init__(self, parent, preferences=None,*a, **k):
        super(MappingComponent, self).__init__(*a, **k)
        self.parent_instance = parent
        self.preferences = preferences or {}
        self.control_mappings = {
            "play": None,
            "stop": None,
            "previous": None,
            "next": None
        }

    def add_mapping(self, control_name, msg_type, channel, identifier):
        """Update the mapping and trigger assignment."""
        if control_name not in self.control_mappings:
            self.parent_instance._log("Unknown control name: {}".format(control_name))
            return

        else:
            self.control_mappings[control_name] = {
                "msg_type": msg_type,
                "channel": channel,
                "identifier": identifier,
            }
            
        self.parent_instance.pending_mappings.append({
            "control_name": control_name,
            "msg_type": msg_type,
            "channel": channel,
            "identifier": identifier,
        })
        self.parent_instance._log("Current Queue: {}".format(self.parent_instance.pending_mappings))
        self.parent_instance.add_map_task()

    def remove_mapping(self, function_name):
        """Removes an existing mapping."""
        if function_name in self.control_mappings:
            self.control_mappings[function_name] = None
            self.parent_instance._log("Removed mapping: {}".format(function_name))
        else:
            self.parent_instance._log("Mapping {} not found.".format(function_name))


    def store(self, preferences):
        """Store the control mappings into preferences."""
        # Ensure mappings are serializable
        serializable_mappings = {
            control_name: {
                "msg_type": mapping["msg_type"],
                "channel": mapping["channel"],
                "identifier": mapping["identifier"]
            } for control_name, mapping in self.control_mappings.items() if mapping
        }
        preferences["control_mappings"] = serializable_mappings
        self.parent_instance._log("Control mappings stored successfully.")

    def restore(self, preferences):
        """Restore the control mappings from preferences."""
        stored_mappings = preferences.get("control_mappings", {})
        if not stored_mappings:
            self.parent_instance._log("No stored control mappings found.")
            return
        
        for control_name, mapping in stored_mappings.items():
            self.control_mappings[control_name] = mapping
            try:
                restored_msg_type = mapping["msg_type"]
                restored_channel = mapping["channel"]
                restored_identifier = mapping["identifier"]
                # Assign the control dynamically
                self.parent_instance.assign_control(
                    control_name, restored_msg_type, restored_channel, restored_identifier
                )
                self.parent_instance._log("Restored mapping: {} -> msg_type={}, channel={}, identifier={}".format(control_name, restored_msg_type, restored_channel, restored_identifier))
            except KeyError as e:
                self.parent_instance._log("Error restoring mapping for {}: Missing key {}".format(control_name, e))
